/******************************************************************************
 *                                                                            *
 *                                Copyright by                                *
 *                                                                            *
 *                              Azoteq (Pty) Ltd                              *
 *                          Republic of South Africa                          *
 *                                                                            *
 *                           Tel: +27(0)21 863 0033                           *
 *                           E-mail: info@azoteq.com                          *
 *                                                                            *
 * ========================================================================== *
 * @file        IQS397.h                                                      *
 * @brief       IQS397 Arduino library header file.                           *
 *                                                                            *
 * @details     This file contains the declarations, definitions, and         *
 *              data structures required to interface with the Azoteq        *
 *              IQS397 device using Arduino. The library provides             *
 *              functionality for device initialization, configuration,      *
 *              and runtime operation.                                        *
 *                                                                            *
 * @author      Azoteq PTY Ltd                                                *
 * @version     v1.0                                                          *
 * @date        2026                                                          *
 * ========================================================================== *
 * @attention   Makes use of the following standard Arduino libraries:        *
 *              - Arduino.h  (included, standard Arduino library)             *
 *              - Wire.h     (included, standard Arduino library)             *
 ******************************************************************************/

#ifndef IQS397_h
#define IQS397_h

// Include Files
#include "Arduino.h"
#include "Wire.h"
#include "IQS397_init_1.h"
#include "./inc/IQS397_addresses.h"

// Public Global Definitions
/* For use with Wire.h library. True argument with some functions closes the
I2C communication window. */
#define STOP true
/* For use with Wire.h library. False argument with some functions keeps the
I2C communication window open. */
#define RESTART false
#define IQS397_I2C_RETRY 10

// Device Info
#define IQS397_PRODUCT_NUM_0 3072
#define IQS397_PRODUCT_NUM_1 3073
#define IQS397_PRODUCT_NUM_2 3074

// System Commands Bits (Register 0x1000)
#define IQS397_ACK_RESET_BIT 0
#define IQS397_SW_RESET_BIT 1
#define IQS397_FORCE_ATI_CH0_BIT 2
#define IQS397_RESEED_CH0 4
#define IQS397_TRIGGER_HAPTICS_BIT 5

//I2C Setings Bits (Register 0x1001)
#define IQS397_POWER_MODE_SETTINGs_BIT 0
#define IQS397_INTERFACE_SELECT_BIT_0 5
#define IQS397_INTERFACE_SELECT_BIT_1 6
#define IQS397_TERMINATE_COMMS_ON_STOP_BIT 7

// Global Event Mask Bits (Register 0x1002)
#define IQS397_POWER_MODE_EVENT_BIT 0
#define IQS397_ATI_EVENT_BIT 1
#define IQS397_PROX_EVENT_BIT 2
#define IQS397_TOUCH_EVENT_BIT 3
#define IQS397_HAPTICS_EVENT_BIT 4

// Drive Settings Bit (Register 0x1049)
#define IQS397_AMPLITUDE_BIT_0 6
#define IQS397_AMPLITUDE_BIT_1 7


// Power Mode Flags Bits (Register 0x2000)
#define IQS397_POWER_MODE_BIT_0 0
#define IQS397_POWER_MODE_BIT_1 1

// Device Status Bit (Register 0x2001)
#define IQS397_SHOW_RESET_BIT 0

// System Event Flags (0x2002)
#define IQS397_PROXIMITY_EVENT_BIT 2
#define IQS397_HAPTICS_EVENT_BIT 4

// Button Event Flags (Registers 0x2005)
#define IQS397_LTA_HALT_STATE_BIT 0
#define IQS397_PROXIMITY_EVENT_STATE_BIT 1
#define IQS397_TOUCH_EVENT_STATE_BIT 2


/* Defines and structs for IQS397 states */
/**
 * @brief  IQS397 Init Enumeration.
 */
typedef enum
{
    IQS397_INIT_NONE,
    IQS397_INIT_VERIFY_PRODUCT,
    IQS397_INIT_ACTIVATE_STREAM_MODE,
    IQS397_INIT_READ_RESET,
    IQS397_INIT_CHIP_RESET,
    IQS397_INIT_UPDATE_SETTINGS,
    IQS397_INIT_CHECK_RESET,
    IQS397_INIT_ACK_RESET,
    IQS397_INIT_ATI,
    IQS397_INIT_WAIT_FOR_ATI,
    IQS397_INIT_READ_DATA,
    IQS397_INIT_ACTIVATE_EVENT_MODE,
    IQS397_INIT_DONE
} IQS397_init_e;

typedef enum
{
    IQS397_STATE_NONE,
    IQS397_STATE_START,
    IQS397_STATE_INIT,
    IQS397_STATE_CHECK_RESET,
    IQS397_STATE_RUN,
} IQS397_state_e;

typedef enum
{
    IQS397_NORMAL_POWER = 0,
    IQS397_ULP = 1,
} IQS397_power_modes;

typedef struct
{
    IQS397_state_e state;
    IQS397_init_e init_state;
} IQS397_s;


/* IQS397 Memory map data variables, only save the data that might be used
during program runtime */
typedef struct
{
    uint8_t power_mode_flags;       // 	0x2000
    uint8_t device_status_flags;    // 0x2001
    uint8_t system_event_flags;      // 	0x2002
    uint8_t prox_states;         // 	0x2003
    uint8_t button_flags;        // 	0x2005
    uint8_t inverted_counts[2]; // 	0x2008
    uint8_t drive_settings_flags;        // 	0x1049


} IQS397_MEMORY_MAP;

// Class Prototype
class IQS397
{
public:
    // Public Constructors
    IQS397();

    // Public Device States
    IQS397_s iqs397_state;

    // Public Variables
    IQS397_MEMORY_MAP IQSMemoryMap;
    bool new_data_available;
    uint16_t previous_interval;
    bool do_soft_reset;

    // Public Methods
    void begin(uint8_t deviceAddressIn, uint8_t readyPinIn, void (*interrupt_func)());
    bool init(void);
    void run(void);
    void queueValueUpdates(void);
    uint16_t getProductNum(bool stopOrRestart);
    uint8_t getmajorVersion(bool stopOrRestart);
    uint8_t getminorVersion(bool stopOrRestart);
    void acknowledgeReset(bool stopOrRestart);
    void ReATI(bool stopOrRestart);
    void SW_Reset(bool stopOrRestart);
    void writeMM(bool stopOrRestart);
    void clearRDY(void);
    bool getRDYStatus(void);

    void setStreamMode(bool stopOrRestart);
    void setEventMode(bool stopOrRestart);

    void updateInfoFlags(bool stopOrRestart);
    IQS397_power_modes get_PowerMode(void);
    bool checkReset(void);

    void triggerHaptics(bool stopOrRestart);
    // bool hapticsConfig(void);

    bool button_ltaHaltState(void);
    bool button_proximityState(void);
    bool button_touchState(void);
    bool haptics_triggerEventState(void);
    bool interface_select(void);

    bool proximityEvent(void);
    bool amplitude_100_driveSettings(bool stopOrRestart);
    bool amplitude_50_driveSettings(bool stopOrRestart);


    void force_I2C_communication(void);
    void ready_interrupt(void);


    void force_Haptics_Trigger(void);


private:
    // Private Variables
    uint8_t _deviceAddress;
    uint8_t _readyPin;
    bool _deviceRDY;

    // Private Methods
    void readRandomBytes(uint16_t memoryAddress, uint8_t numBytes, uint8_t bytesArray[], bool stopOrRestart);
    void writeRandomBytes(uint16_t memoryAddress, uint8_t numBytes, uint8_t bytesArray[], bool stopOrRestart);
    bool getBit(uint8_t data, uint8_t bit_number);
    uint8_t setBit(uint8_t data, uint8_t bit_number);
    uint8_t clearBit(uint8_t data, uint8_t bit_number);
};
#endif // IQS397_h
