/******************************************************************************
 *                                                                            *
 *                                Copyright by                                *
 *                                                                            *
 *                              Azoteq (Pty) Ltd                              *
 *                          Republic of South Africa                          *
 *                                                                            *
 *                           Tel: +27(0)21 863 0033                           *
 *                           E-mail: info@azoteq.com                          *
 *                                                                            *
 * ========================================================================== *
 * @file        IQS397.cpp                                                    *
 * @brief       IQS397 Arduino library source file.                           *
 *                                                                            *
 * @details     This file contains the constructors and methods used to       *
 *              initialize, configure, and operate the Azoteq IQS397 device   *
 *              from an Arduino platform. The implementation provides        *
 *              functionality for inductive/capacitive touch, proximity, haptics,      *
 *              and power-mode handling via an I2C interface.                *
 *                                                                            *
 * @author      Azoteq PTY Ltd                                                *
 * @version     v1.0                                                          *
 * @date        2026                                                          *
 * ========================================================================== *
 * @attention   Makes use of the following standard Arduino libraries:        *
 *              - Arduino.h  (included via IQS397.h)                          *
 *              - Wire.h     (included via IQS397.h)                          *
 ******************************************************************************/

/* Include Files */
#include "IQS397.h"

/******************************************************************************/
/*                                CONSTRUCTORS                                */
/******************************************************************************/
IQS397::IQS397()
{
}

/******************************************************************************/
/*                               PUBLIC METHODS                               */
/******************************************************************************/

/**
 * @name    begin
 * @brief   A method to initialize the IQS397 device with the device address
 *          and ready pin specified by the user.
 * @param   deviceAddress -> The address of the IQS397 device.
 * @param   readyPin      -> The Arduino pin which is connected to the ready
 *                           pin of the IQS397 device. Should be an
 *                           interrupt-capable pin.
 * @param   interrupt_func -> Pointer to function that must be executed on the
 *                            ready interrupt.
 *                            This is a user-defined function that should call
 *                            iqs397.ready_interrupt().
 * @retval  None.
 */
void IQS397::begin(uint8_t deviceAddressIn, uint8_t readyPinIn, void (*interrupt_func)())
{
    // Initialize I2C communication here, since this library can't function without it.
    Wire.begin();
    Wire.setClock(400000);

    _deviceAddress = deviceAddressIn;
    _readyPin = readyPinIn;
    _deviceRDY = false;
    do_soft_reset = false;
    attachInterrupt(digitalPinToInterrupt(_readyPin), interrupt_func, CHANGE);

    /* Initialize "running" and "init" state machine variables. */
    iqs397_state.state = IQS397_STATE_START;
    iqs397_state.init_state = IQS397_INIT_VERIFY_PRODUCT;
}

/**
 * @name    init
 * @brief   A method that runs through a normal start-up routine to set up the
 *          IQS397 with the desired settings from the IQS397_init-1.h file.
 * @details This function must be called repeatedly until it returns true.
 * @retval  Returns true if the full start-up routine has been completed,
 *          returns false if not.
 * @note    No false return will be given, the program will thus be stuck when
 *          one of the cases is not able to finish.
 *          See serial communication to find the ERROR case
 */
bool IQS397::init(void)
{
    uint16_t prod_num;
    uint8_t ver_maj, ver_min;

    switch (iqs397_state.init_state)
    {
    /* Verifies product number to determine if the correct device is connected for
    this example */
    case IQS397_INIT_VERIFY_PRODUCT:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_VERIFY_PRODUCT");
            prod_num = getProductNum(RESTART);
            ver_maj = getmajorVersion(RESTART);
            ver_min = getminorVersion(STOP);
            Serial.print("\t\tProduct number is: ");
            Serial.print(prod_num);
            Serial.print(" v");
            Serial.print(ver_maj);
            Serial.print(".");
            Serial.println(ver_min);
            if ((prod_num == IQS397_PRODUCT_NUM_0) ||
                (prod_num == IQS397_PRODUCT_NUM_1) ||
                (prod_num == IQS397_PRODUCT_NUM_2))
            {
                Serial.println("\t\tIQS397 Confirmed!");
                iqs397_state.init_state = IQS397_INIT_ACTIVATE_STREAM_MODE;
            }
            else
            {
                Serial.println("\t\tDevice is not an IQS397!");
                iqs397_state.init_state = IQS397_INIT_NONE;
            }
        }
        break;

    /* Start I2C streaming mode */
    case IQS397_INIT_ACTIVATE_STREAM_MODE:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_ACTIVATE_STREAM_MODE");
            setStreamMode(STOP);
            iqs397_state.init_state = IQS397_INIT_READ_RESET;
        }
        break;


    /* Verify if a reset has occurred */
    case IQS397_INIT_READ_RESET:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_READ_RESET");
            updateInfoFlags(RESTART);
            if (checkReset())
            {
                Serial.println("\t\tReset event occurred.");
                iqs397_state.init_state = IQS397_INIT_ACK_RESET;
            }
            else
            {
                Serial.println("\t\t No Reset Event Detected - Request SW Reset");
                iqs397_state.init_state = IQS397_INIT_CHIP_RESET;
            }
        }
        break;

    /* Perform SW Reset */
    case IQS397_INIT_CHIP_RESET:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_CHIP_RESET");

            // Perform SW Reset
            SW_Reset(STOP);
            Serial.println("\t\tSoftware Reset Bit Set.");
            delay(100);
            iqs397_state.init_state = IQS397_INIT_READ_RESET;
        }
        break;

    /* Write all settings to IQS397 from .h file */
    case IQS397_INIT_UPDATE_SETTINGS:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_UPDATE_SETTINGS");
            writeMM(STOP);
            iqs397_state.init_state = IQS397_INIT_ATI;
        }
        break;

    /* Acknowledge that the device went through a reset */
    case IQS397_INIT_ACK_RESET:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_ACK_RESET");
            acknowledgeReset(STOP);
            iqs397_state.init_state = IQS397_INIT_UPDATE_SETTINGS;
        }
        break;

    /* Run the ATI algorithm to recalibrate the device with newly added settings */
    case IQS397_INIT_ATI:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_ATI");
            ReATI(STOP);
            iqs397_state.init_state = IQS397_INIT_READ_DATA;
        }
        break;

    /* Read the latest data from the iqs397 */
    case IQS397_INIT_READ_DATA:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_READ_DATA");
            queueValueUpdates();
            iqs397_state.init_state = IQS397_INIT_ACTIVATE_EVENT_MODE;
        }
        break;

    /* Turn on I2C Event mode */
    case IQS397_INIT_ACTIVATE_EVENT_MODE:
        if (_deviceRDY)
        {
            Serial.println("\tIQS397_INIT_ACTIVATE_EVENT_MODE");
            setEventMode(STOP);
            iqs397_state.init_state = IQS397_INIT_DONE;
        }
        break;

    /* All operations have been completed correctly */
    case IQS397_INIT_DONE:
        Serial.println("\tIQS397_INIT_DONE");
        new_data_available = true;
        return true;
        break;

    default:
        break;
    }
    return false;
}

/**
 * @name    run
 * @brief   Run the device ready check and read the necessary bytes when the
 *          IQS397 has pulled the RDY line low.
 *          The new_data_available flag will be set when a ready low is received
 *          from the IQS397.
 * @note    queueValueUpdates can be edited by the user if other data should be
 *          read every time a RDY window is received.
 */
void IQS397::run(void)
{
    switch (iqs397_state.state)
    {
    /* After a hardware reset, this is the starting position of the main running
    state machine */
    case IQS397_STATE_START:
        Serial.println("IQS397 Initialization:");
        iqs397_state.state = IQS397_STATE_INIT;
        break;

    /* Perform the initialization routine on the IQS397 */
    case IQS397_STATE_INIT:
        if (init())
        {
            Serial.println("IQS397 Initialization complete!\n");
            iqs397_state.state = IQS397_STATE_CHECK_RESET;
        }
        break;

    /* Continuous reset monitoring state, ensure no reset event has occurred
    for data to be valid */
    case IQS397_STATE_CHECK_RESET:
        if (checkReset())
        {
            Serial.println("Reset Occurred!\n");
            new_data_available = false;
            iqs397_state.state = IQS397_STATE_START;
            iqs397_state.init_state = IQS397_INIT_VERIFY_PRODUCT;
        }
        /* A reset did not occur, move to the run state and wait for a new ready
        window */
        else
        {
            new_data_available = true; /* No reset, thus data is valid */
            iqs397_state.state = IQS397_STATE_RUN;
        }
        break;

    /* If a RDY Window is open, read the latest values from the IQS397 */
    case IQS397_STATE_RUN:
        new_data_available = false;
        if (_deviceRDY)
        {
            if (do_soft_reset)
            {
                do_soft_reset = false;
                SW_Reset(STOP);
                return;
            }

            queueValueUpdates();
            _deviceRDY = false;
            iqs397_state.state = IQS397_STATE_CHECK_RESET;
        }
        break;

    default:
        break;
    }
}

/**
 * @name    iqs397_ready_interrupt
 * @brief   A method used as an interrupt function. Only activated when a falling-
 *          edge interrupt is seen on the correct Arduino interrupt pin.
 * @note    Keep this function as simple as possible to prevent stuck states and
 *          slow operations.
 */
void IQS397::ready_interrupt(void)
{
    if (digitalRead(_readyPin))
    {
        _deviceRDY = false;
    }
    else
    {
        _deviceRDY = true;
    }
}

/**
 * @name   clearRDY
 * @brief  A method used to clear the ready interrupt bit.
 */
void IQS397::clearRDY(void)
{
    _deviceRDY = false;
}

/**
 * @name   getRDYStatus
 * @brief  A method used to retrieve the device RDY status.
 * @param  None.
 * @retval Returns the boolean IQS397 RDY state.
 *         - True when RDY line is LOW
 *         - False when RDY line is HIGH
 */
bool IQS397::getRDYStatus(void)
{
    return _deviceRDY;
}

/**
 * @name    queueValueUpdates
 * @brief   All I2C read operations in the queueValueUpdates method will be
 *          performed each time the IQS397 opens a RDY window.
 * @note    Any Address in the memory map can be read from here.
 */
void IQS397::queueValueUpdates(void)
{

    uint8_t transferBytes[10]; // The array which will hold the bytes to be transferred.

    /* Read the information necessary for this specific implementation from the
    IQS397 */
    readRandomBytes(IQS397_MM_SYSTEM_FLAGS, 6, transferBytes, RESTART);

    /* Assign the system status flags to the local device_status variable */
    IQSMemoryMap.power_mode_flags = transferBytes[0];
    IQSMemoryMap.device_status_flags = transferBytes[1];
    IQSMemoryMap.system_event_flags = transferBytes[2];

    /* Assign the events flags to the local events variables */
    IQSMemoryMap.prox_states = transferBytes[3];

    IQSMemoryMap.button_flags = transferBytes[5];

    if (proximityEvent())
    {
        amplitude_50_driveSettings(RESTART);
        triggerHaptics(STOP);
    }
    else
    {
        amplitude_100_driveSettings(STOP);
    }

}

void IQS397::triggerHaptics(bool stopOrRestart)
{
    uint8_t transferBytes[1]; // A temporary array to hold the bytes transferred.

    // Read the System Flags from the IQS397
    // These must be read first in order not to change any settings.
    readRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferBytes, RESTART);
    // Write the Ack Reset bit to 1 to clear the Show Reset Flag.
    transferBytes[0] = setBit(transferBytes[0], IQS397_TRIGGER_HAPTICS_BIT);
    // Write the new byte to the System Flags address.
    writeRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferBytes, stopOrRestart);
}

/**
 * @name	checkReset
 * @brief   A method that checks if the device has reset and returns the reset
 *          status.
 * @retval  Returns true if a reset has occurred, false if no reset has
 *          occurred.
 * @note    If a reset has occurred the device settings should be reloaded using
 *          the begin function. After new device settings have been reloaded the
 *          acknowledge reset function can be used to clear the reset flag.
 */
bool IQS397::checkReset(void)
{
    /* Perform a bitwise AND operation inside getBIT with the DEVICE_RESET_BIT
   to return the reset status */
    return getBit(IQSMemoryMap.device_status_flags, IQS397_SHOW_RESET_BIT);
}

/**
 * @name	checkProductNum
 * @brief   Read and return the device product number.
 * @param   stopOrRestart -> Specifies whether the communications window must
 *                           be kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @retval  Returns the product number as a 16-bit unsigned integer value.
 * @note    If the product is not correctly identified an appropriate messages
 *          should be displayed.
 */
uint16_t IQS397::getProductNum(bool stopOrRestart)
{
    uint8_t transferBytes[2]; // A temporary array to hold the read bytes
    uint16_t prodNumReturn;   // The 16-bit return value.

    // Read the Device info from the IQS397.
    readRandomBytes(IQS397_MM_PROD_NUM, 2, transferBytes, stopOrRestart);

    // Construct the 16-bit return value.
    prodNumReturn = (uint16_t)(transferBytes[0]);
    prodNumReturn |= (uint16_t)(transferBytes[1] << 8);

    // Return the product number
    return prodNumReturn;
}

/**
 * @name	getmajorVersion
 * @brief   A method that reads the device firmware major version number.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @retval Returns the major version number as an 8-bit unsigned integer value.
 */
uint8_t IQS397::getmajorVersion(bool stopOrRestart)
{
    uint8_t transferBytes[2]; // A temporary array to hold the read bytes

    // Read the Device info from the IQS397.
    readRandomBytes(IQS397_MM_MAJOR_VERSION_NUM, 2, transferBytes, stopOrRestart);

    // Return the major firmware version number value.
    return transferBytes[0];
}

/**
 * @name	getminorVersion
 * @brief   A method that reads the device firmware minor version number.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @retval Returns the minor version number as an 8-bit unsigned integer value.
 */
uint8_t IQS397::getminorVersion(bool stopOrRestart)
{
    uint8_t transferBytes[2]; // A temporary array to hold the read bytes

    // Read the Device info from the IQS397.
    readRandomBytes(IQS397_MM_MINOR_VERSION_NUM, 2, transferBytes, stopOrRestart);

    // Return the minor firmware version number value.
    return transferBytes[0];
}


/**
 * @name	acknowledgeReset
 * @brief   A method that clears the Show Reset flag by setting the ACK_RESET
 *          bit.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    If a reset has occurred the device settings should be reloaded using
 *          the begin function. After new device settings have been reloaded,
 *          this method should be used to clear the reset bit.
 * @note    To acknowledge reset, IQS397_ACK_RESET_BIT of SYSTEM SETTINGS
 *          is set.
 */
void IQS397::acknowledgeReset(bool stopOrRestart)
{
    uint8_t transferBytes[2]; // A temporary array to hold the bytes transferred.

    // Read the System Flags from the IQS397
    // These must be read first in order not to change any settings.
    readRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 2, transferBytes, RESTART);
    // Write the Ack Reset bit to 1 to clear the Show Reset Flag.
    transferBytes[0] = setBit(transferBytes[0], IQS397_ACK_RESET_BIT);
    // Write the new byte to the System Flags address.
    writeRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 2, transferBytes, stopOrRestart);
}

/**
 * @name    ReATI
 * @brief   A method that sets the REDO_ATI_BIT in order to force the IQS397
 *          device to run the Automatic Tuning Implementation (ATI) routine.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    To perform ATI, bits 2 and 3 of SYSTEM SETTINGS is set.
 */
void IQS397::ReATI(bool stopOrRestart)
{
    uint8_t transferByte[1]; // Array to store the bytes transferred.

    readRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferByte, RESTART);
    // Mask the settings with the REDO_ATI_BIT.
    transferByte[0] = setBit(transferByte[0], IQS397_FORCE_ATI_CH0_BIT);
    // Write the new byte to the required device.
    writeRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferByte, stopOrRestart);
}

/**
 * @name    SW_Reset
 * @brief   A method that sets the SW RESET bit to force the IQS397
 *          device to do a SW reset.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    To perform SW Reset, bit 1 in SYSTEM SETTINGS is set.
 */
void IQS397::SW_Reset(bool stopOrRestart)
{
    uint8_t transferByte[1]; // Array to store the bytes transferred.

    readRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferByte, RESTART);
    // Mask the settings with the SW_RESET_BIT.
    transferByte[0] = setBit(transferByte[0], IQS397_SW_RESET_BIT);
    // Write the new byte to the required device.
    writeRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 1, transferByte, stopOrRestart);
}

/**
 * @name    setEventMode
 * @brief   A method to set the IQS397 device into event mode.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    To enter event mode, IQS397_INTERFACE_SELECT_BIT of SYSTEM SETTINGS
 *          is set.
 */
void IQS397::setEventMode(bool stopOrRestart)
{
    uint8_t transferBytes[1]; // The array which will hold the bytes which are transferred.

    // First read the bytes at the memory address so that they can be preserved.
    readRandomBytes(IQS397_MM_SYSTEM_SETTINGS, 1, transferBytes, RESTART);

    // Set the EVENT_MODE_BIT in UI_SETTINGS
    transferBytes[0] = setBit(transferBytes[0], IQS397_INTERFACE_SELECT_BIT_0);
    transferBytes[0] = clearBit(transferBytes[0], IQS397_INTERFACE_SELECT_BIT_1);
    // Write the bytes back to the device
    writeRandomBytes(IQS397_MM_SYSTEM_SETTINGS, 1, transferBytes, stopOrRestart);
}

/**
 * @name    setStreamMode
 * @brief   A method to set the IQS397 device into streaming mode.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    To exit event mode, IQS397_INTERFACE_SELECT_BIT of SYSTEM SETTINGS
 *          is cleared.
 */
void IQS397::setStreamMode(bool stopOrRestart)
{
    uint8_t transferBytes[1]; // The array which will hold the bytes transferred.

    // First read the bytes at the memory address so that they can be preserved.
    readRandomBytes(IQS397_MM_SYSTEM_SETTINGS, 1, transferBytes, RESTART);
    // Clear the EVENT_MODE bit in SYSTEM SETTINGS
    transferBytes[0] = clearBit(transferBytes[0], IQS397_INTERFACE_SELECT_BIT_0);
    transferBytes[0] = clearBit(transferBytes[0], IQS397_INTERFACE_SELECT_BIT_1);
    // Write the bytes back to the device
    writeRandomBytes(IQS397_MM_SYSTEM_SETTINGS, 1, transferBytes, stopOrRestart);
}

/**
 * @name    updateInfoFlags
 * @brief   A method which reads the IQS397 info flags and assigns them to the
 *          local IQSMemoryMap struct.
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    The local `device_status_flags` and `device_status` values are altered with
 *          the new value of the info flags registers retrieved from the IQS397.
 */
void IQS397::updateInfoFlags(bool stopOrRestart)
{
    uint8_t transferBytes[2]; // The array which will hold the bytes to be transferred.

    // Read the info flags.
    readRandomBytes(IQS397_MM_SYSTEM_FLAGS, 2, transferBytes, stopOrRestart);
    // Assign the info flags to the local SYSTEM_STATUS register.
    IQSMemoryMap.power_mode_flags = transferBytes[0];
    IQSMemoryMap.device_status_flags = transferBytes[1];
}

/**
 * @name    get_PowerMode
 * @brief   A method that reads the local `device_status_flags` register and returns
 *          the current power mode.
 * @retval  Returns the iqs397_power_modes state the device is in.
 * @note    See Datasheet on power mode options and timeouts.
 *          High Accuracy, Normal Power, Low Power, Ultra Low Power (ULP).
 */
IQS397_power_modes IQS397::get_PowerMode(void)
{
    uint8_t buffer = getBit(IQSMemoryMap.power_mode_flags, IQS397_POWER_MODE_BIT_0);
    buffer += getBit(IQSMemoryMap.power_mode_flags, IQS397_POWER_MODE_BIT_1) << 1;

    return (IQS397_power_modes)buffer;
}

/**
 * @name    button_ltaHaltState
 * @brief   A method that checks the local `button_flags` if the button is
 *          in active state.
 * @retval  Returns true if a active is detected and false if there is no active.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::button_ltaHaltState(void)
{
    return getBit(IQSMemoryMap.button_flags, IQS397_LTA_HALT_STATE_BIT);
}

/**
 * @name    button_proximityState
 * @brief   A method that checks the local `movement_flags` if the button is
 *          in a movement enter state.
 * @retval  Returns true if a enter is detected and false if there is no enter.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::button_proximityState(void)
{
    return getBit(IQSMemoryMap.button_flags, IQS397_PROXIMITY_EVENT_STATE_BIT);
}

/**
 * @name    button_touchState
 * @brief   A method that checks the local `button_flags` if the button is
 *          in a movement exit state.
 * @retval  Returns true if a exit is detected and false if there is no exit.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::button_touchState(void)
{
    return getBit(IQSMemoryMap.button_flags, IQS397_TOUCH_EVENT_STATE_BIT);
}

/**
 * @name    haptics_triggerEventState
 * @brief   A method that checks the local `system_event_flags` if the button is
 *          in a movement active state.
 * @retval  Returns true if a movement is detected and false if there is no movement.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::haptics_triggerEventState(void)
{
    return getBit(IQSMemoryMap.system_event_flags, IQS397_HAPTICS_EVENT_BIT);
}

/**
 * @name    proximityEvent
 * @brief   A method that checks the local `system_event_flags` if the button is
 *          in a proximity event.
 * @retval  Returns true if a movement is detected and false if there is no movement.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::proximityEvent(void)
{
    return getBit(IQSMemoryMap.system_event_flags, IQS397_PROXIMITY_EVENT_BIT);
}

/**
 * @name    amplitude_100_driveSettings
 * @brief   A method that checks the local `drive_settings_flags` if the button is
 *          in a proximity event.
 * @retval  Returns true if a movement is detected and false if there is no movement.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::amplitude_100_driveSettings(bool stopOrRestart)
{
    uint8_t transferBytes[1]; // The array which will hold the bytes which are transferred.

    // First read the bytes at the memory address so that they can be preserved.
    readRandomBytes(IQS397_MM_DRIVE_SETTINGS, 1, transferBytes, RESTART);

    // Set the AMPLITUDE_BIT in UI_SETTINGS
    transferBytes[0] = setBit(transferBytes[0], IQS397_AMPLITUDE_BIT_0);
    transferBytes[0] = setBit(transferBytes[0], IQS397_AMPLITUDE_BIT_1);
    // Write the bytes back to the device
    writeRandomBytes(IQS397_MM_DRIVE_SETTINGS, 1, transferBytes, stopOrRestart);
}

/**
 * @name    amplitude_50_driveSettings
 * @brief   A method that checks the local `drive_settings_flags` if the button is
 *          in a proximity event.
 * @retval  Returns true if a movement is detected and false if there is no movement.
 * @note    The IQS397 only has 1 touch output channel.
 */
bool IQS397::amplitude_50_driveSettings(bool stopOrRestart)
{
    uint8_t transferBytes[1]; // The array which will hold the bytes which are transferred.

    // First read the bytes at the memory address so that they can be preserved.
    readRandomBytes(IQS397_MM_DRIVE_SETTINGS, 1, transferBytes, RESTART);

    // Set the AMPLITUDE_BIT in UI_SETTINGS
    transferBytes[0] = clearBit(transferBytes[0], IQS397_AMPLITUDE_BIT_0);
    transferBytes[0] = setBit(transferBytes[0], IQS397_AMPLITUDE_BIT_1);
    // Write the bytes back to the device
    writeRandomBytes(IQS397_MM_DRIVE_SETTINGS, 1, transferBytes, stopOrRestart);
}

/******************************************************************************/
/*                           ADVANCED PUBLIC METHODS                          */
/******************************************************************************/

/**
 * @name    writeMM
 * @brief   Function to write the whole memory map to the device (writable)
 *          registers
 * @param   stopOrRestart -> Specifies whether the communications window must be
 *                           kept open or must be closed after this action.
 *                           Use the STOP and RESTART definitions.
 * @note    Requires the `IQS397_init-1.h` file with the defined settings.
 */
void IQS397::writeMM(bool stopOrRestart)
{
    // Temporary array which holds the bytes to be transferred.
    uint8_t transferBytes[32];

    /* System Control Settings */
    /* Memory Map Position 0x1000 - 0x1011 */
    transferBytes[0] = AUTO_CLEAR_SETTINGS;
    transferBytes[1] = SYSTEM_SETTINGS;
    transferBytes[2] = EVENTMASK;
    transferBytes[3] = ULP_SETTINGS;
    transferBytes[4] = AUTO_ULP_THRESHOLD_0;
    transferBytes[5] = AUTO_ULP_THRESHOLD_1;
    transferBytes[6] = NP_REPORT_0;
    transferBytes[7] = NP_REPORT_1;
    transferBytes[8] = ULP_REPORT_0;
    transferBytes[9] = ULP_REPORT_1;
    transferBytes[10] = EMPTY_0;
    transferBytes[11] = EMPTY_1;
    transferBytes[12] = EMPTY_2;
    transferBytes[13] = EMPTY_3;
    transferBytes[14] = ULP_TIMEOUT_0;
    transferBytes[15] = ULP_TIMEOUT_1;
    transferBytes[16] = I2C_TIMEOUT_0;
    transferBytes[17] = I2C_TIMEOUT_1;

    writeRandomBytes(IQS397_MM_SYSTEM_COMMANDS, 18, transferBytes, RESTART);
    Serial.println("\t\t1. Write System Settings");


    /* Channel 0 Settings 0 */
    /* Memory Map Position 0x1012 - 0x1029 */
    transferBytes[0] = CONVERSION_SETUP;
    transferBytes[1] = CONVERSION_SETTINGS;
    transferBytes[2] = BUTTON_PROX_TIMEOUT_0;
    transferBytes[3] = BUTTON_PROX_TIMEOUT_1;
    transferBytes[4] = BUTTON_TOUCH_TIMEOUT_0;
    transferBytes[5] = BUTTON_TOUCH_TIMEOUT_1;
    transferBytes[6] = BUTTON_ATI_TIMEOUT_0;
    transferBytes[7] = BUTTON_ATI_TIMEOUT_1;
    transferBytes[8] = BUTTON_FILTER_COUNT_BETA;
    transferBytes[9] = BUTTON_FILTER_LTA_BETA;
    transferBytes[10] = BUTTON_FILTER_LTA_BETA_FAST;
    transferBytes[11] = BUTTON_FILTER_LTA_ULP_BETA;
    transferBytes[12] = BUTTON_FILTER_LTA_BETA_FAST_BAND_0;
    transferBytes[13] = BUTTON_FILTER_LTA_BETA_FAST_BAND_1;
    transferBytes[14] = EMPTY_4;
    transferBytes[15] = EMPTY_5;
    transferBytes[16] = EMPTY_6;
    transferBytes[17] = EMPTY_7;
    transferBytes[18] = EMPTY_8;
    transferBytes[19] = EMPTY_9;
    transferBytes[20] = BUTTON_PROX_THRESHOLD_0;
    transferBytes[21] = BUTTON_PROX_THRESHOLD_0;
    transferBytes[22] = BUTTON_PROX_DEBOUNCE;
    transferBytes[23] = EMPTY_10;

    writeRandomBytes(IQS397_MM_PF_SETTINGS_0, 24, transferBytes, RESTART);
    Serial.println("\t\t2. Write ProxFusion CH0 Settings");


    /* Channel 0 Settings 1 */
    /* Memory Map Position 0x102a - 0x1031 */
    transferBytes[0] = BUTTON_TOUCH_THRESHOLD_0;
    transferBytes[1] = BUTTON_TOUCH_THRESHOLD_1;
    transferBytes[2] = BUTTON_TOUCH_DEBOUNCE;
    transferBytes[3] = BUTTON_TOUCH_HYST;
    transferBytes[4] = BUTTON_ATI_BAND_0;
    transferBytes[5] = BUTTON_ATI_BAND_1;
    transferBytes[6] = BUTTON_LTA_HALT_THRESHOLD_0;
    transferBytes[7] = BUTTON_LTA_HALT_THRESHOLD_1;


    writeRandomBytes(IQS397_MM_PF_SETTINGS_1, 8, transferBytes, RESTART);
    Serial.println("\t\t3. Write ProxFusion CH1 Settings");

    /* Channel 0 ATI Settings */
    /* Memory Map Position 0x1032 - 0x103D */
    transferBytes[0] = SENSOR0_ATI_BASE_0;
    transferBytes[1] = SENSOR0_ATI_BASE_1;
    transferBytes[2] = SENSOR0_ATI_TARGET_0;
    transferBytes[3] = SENSOR0_ATI_TARGET_1;
    transferBytes[4] = EMPTY_11;
    transferBytes[5] = EMPTY_12;
    transferBytes[6] = EMPTY_13;
    transferBytes[7] = EMPTY_14;
    transferBytes[8] = SENSOR0_COARSE;
    transferBytes[9] = SENSOR0_FINE;
    transferBytes[10] = SENSOR0_COMPENSATION_0;
    transferBytes[11] = SENSOR0_COMPENSATION_1;

    writeRandomBytes(IQS397_MM_PF_ATI_SETTINGS, 12, transferBytes, RESTART);
    Serial.println("\t\t4. Write ProxFusion ATI Settings");


    /* Haptic & H-bridge Configuration */
    /* Memory Map Position 0x103E - 0x1043 */
    transferBytes[0] = HAPTIC_CONTROL_0;
    transferBytes[1] = HAPTIC_CONTROL_1;
    transferBytes[2] = OVER_TEMPERATURE_SETTINGS_0;
    transferBytes[3] = OVER_TEMPERATURE_SETTINGS_1;
    transferBytes[4] = H_BRIDGE_SETUP_0;
    transferBytes[5] = H_BRIDGE_SETUP_1;

    writeRandomBytes(IQS397_MM_HAPTIC_CONFIG_SETTINGS_0, 6, transferBytes, RESTART);
    Serial.println("\t\t5. Write Haptic \& H-bridge Configuration");


    /* LRA Drive Settings */
    /* Memory Map Position 0x1044 - 0x1049 */
    transferBytes[0] = PWM_FREQUENCY_0;
    transferBytes[1] = PWM_FREQUENCY_1;
    transferBytes[2] = LRA_FREQUENCY_0;
    transferBytes[3] = LRA_FREQUENCY_1;
    transferBytes[4] = AUTORESONANCE_BACKOFF;
    transferBytes[5] = DRIVE_SETTINGS;

    writeRandomBytes(IQS397_MM_HAPTIC_CONFIG_SETTINGS_1, 6, transferBytes, stopOrRestart);
    Serial.println("\t\t6. LRA Drive Settings");


}

/******************************************************************************/
/*                               PRIVATE METHODS                              */
/******************************************************************************/

/**
 * @name    readRandomBytes
 * @brief   A method that reads a specified number of bytes from a specified
 *          16-bit address and saves it into a user-supplied array.
 * @param   memoryAddress -> The memory address at which to start reading bytes
 *                           from.
 * @param   numBytes      -> The number of bytes that must be read.
 * @param   bytesArray    -> The array which will store the bytes to be read,
 *                           this array will be overwritten.
 * @param   stopOrRestart -> A boolean that specifies whether the communication
 *                           window should remain open or be closed after transfer.
 *                           False keeps it open, true closes it. Use the STOP
 *                           and RESTART definitions.
 * @retval  No value is returned, however, the user-supplied array is overwritten.
 * @note    Uses standard Arduino "Wire" library which is for I2C communication.
 *          Take note that C++ cannot return an array, therefore, the array which
 *          is passed as an argument is overwritten with the required values.
 *          Pass an array to the method by using only its name, e.g. "bytesArray",
 *          without the brackets, this passes a pointer to the array.
 */

void IQS397::readRandomBytes(uint16_t memoryAddress,
                             uint8_t numBytes,
                             uint8_t bytesArray[],
                             bool stopOrRestart)
{
    int i = 0;
    uint8_t error_s = 0; // I2C Error variable to track any issues during communication

    /* Select the device with the address "_deviceAddress" and start
    communication. */
    Wire.beginTransmission(_deviceAddress);
    // Specify the memory address where the IQS397 must start saving the data,
    // as designated by the "memoryAddress" variable.
    uint16_t addr_h = memoryAddress >> 8;
    uint16_t addr_l = memoryAddress;

    Wire.write(addr_l);
    Wire.write(addr_h);

    error_s = Wire.endTransmission(RESTART); // Restart transmission for reading that follows.
    /* The required device has now been selected and it has been told which register to send information from. */

    /* Request "numBytes" bytes from the device which has address "deviceAddress"*/
    uint8_t counter = 0;
    do
    {
        Wire.requestFrom((int)_deviceAddress, (int)numBytes, (int)stopOrRestart);

        /* break out of request loop if max retry is reached */
        if (counter++ >= IQS397_I2C_RETRY)
        {
            return;
        }
    } while (Wire.available() == 0); // Wait for response, this sometimes takes a few attempts

    /* Load the received bytes into the array until there are no more */
    while (Wire.available())
    {
        /* Load the received bytes into the user supplied array */
        bytesArray[i] = Wire.read();
        i++;
    }

    /* Always manually close the RDY window after a STOP is sent to prevent
    writing while the RDY window closes */
    if (stopOrRestart == STOP)
    {
        _deviceRDY = false;
    }
}

/**
 * @name   writeRandomBytes
 * @brief  A method that writes a specified number of bytes to a specified
 *         16-bit address, the bytes to write are supplied using an array pointer.
 * @param  memoryAddress -> The memory address at which to start writing the
 *         bytes to.
 * @param  numBytes      -> The number of bytes that must be written.
 * @param  bytesArray    -> The array which stores the bytes which will be
 *                          written to the memory location.
 * @param  stopOrRestart -> A boolean that specifies whether the communication
 *                          window should remain open or be closed of transfer.
 *                          False keeps it open, true closes it. Use the STOP
 *                          and RESTART definitions.
 * @retval No value is returned, only the IQS device registers are altered.
 * @note   Uses standard Arduino "Wire" library which is for I2C communication.
 *         Take note that a full array cannot be passed to a function in C++.
 *         Pass an array to the function by using only its name, e.g. "bytesArray",
 *         without the square brackets, this passes a pointer to the
 *         array. The values to be written must be loaded into the array prior
 *         to passing it to the function.
 */
void IQS397::writeRandomBytes(uint16_t memoryAddress, uint8_t numBytes, uint8_t bytesArray[], bool stopOrRestart)
{
    uint8_t error_s = 0; // I2C Error variable to track any issues during communication

    /* Select the device with the address of "deviceAddress" and start communication. */
    Wire.beginTransmission(_deviceAddress);

    /* Specify the memory address */
    uint16_t addr_h = memoryAddress >> 8;
    uint16_t addr_l = memoryAddress;

    Wire.write(addr_l);
    Wire.write(addr_h);

    /* Write the bytes as specified in the array. */
    for (int i = 0; i < numBytes; i++)
    {
        Wire.write(bytesArray[i]);
    }
    /* End the transmission, user decides to STOP or RESTART. */
    error_s = Wire.endTransmission(stopOrRestart);

    /* Always manually close the RDY window after a STOP is sent to prevent
    writing while the RDY window closes */
    if (stopOrRestart == STOP)
    {
        _deviceRDY = false;
    }
}

/**
 * @name   getBit
 * @brief  A method that returns the chosen bit value of the provided byte.
 * @param  data       -> byte of which a given bit value needs to be calculated.
 * @param  bit_number -> a number between 0 and 7 representing the bit in question.
 * @retval The boolean value of the specific bit requested.
 */
bool IQS397::getBit(uint8_t data, uint8_t bit_number)
{
    return (data & (1 << bit_number)) >> bit_number;
}

/**
 * @name   setBit
 * @brief  A method that sets the chosen bit value of the provided byte.
 * @param  data       -> byte of which a given bit value needs to be calculated.
 * @param  bit_number -> a number between 0 and 7 representing the bit in question.
 * @retval Returns an 8-bit unsigned integer value of the given data byte with
 *         the requested bit set.
 */
uint8_t IQS397::setBit(uint8_t data, uint8_t bit_number)
{
    return (data |= 1UL << bit_number);
}

/**
 * @name   clearBit
 * @brief  A method that clears the chosen bit value of the provided byte.
 * @param  data       -> byte of which a given bit value needs to be calculated.
 * @param  bit_number -> a number between 0 and 7 representing the bit in question.
 * @retval Returns an 8-bit unsigned integer value of the given data byte with
 *         the requested bit cleared.
 */
uint8_t IQS397::clearBit(uint8_t data, uint8_t bit_number)
{
    return (data &= ~(1UL << bit_number));
}

/**
 * @name   force_I2C_communication
 * @brief  A method that writes 0xFF to open a communication window on the
 *         IQS397.
 * @note   Uses standard Arduino "Wire" library which is for I2C communication.
 */
void IQS397::force_I2C_communication(void)
{
    /*Ensure RDY is HIGH at the moment*/
    if (digitalRead(_readyPin))
    {
        /* Select the device with the address: "DEMO_IQS397_ADDR" and start
        communication. */
        Wire.beginTransmission(_deviceAddress);

        /* Write to memory address 0xFF that will prompt the IQS397 to open a
        communication window.*/
        Wire.write(0xFF);

        /* End the transmission, user decides to STOP or RESTART. */
        Wire.endTransmission(STOP);
    }
}


/**
 * @name   force_Haptics_Trigger
 * @brief  A method that writes 0xFF to open a communication window on the
 *         IQS397.
 * @note   Uses standard Arduino "Wire" library which is for I2C communication.
 */
void IQS397::force_Haptics_Trigger(void)
{
    /*Ensure RDY is HIGH at the moment*/
    if (digitalRead(_readyPin))
    {
        /* Select the device with the address: "DEMO_IQS397_ADDR" and start
        communication. */
        Wire.beginTransmission(_deviceAddress);

        /* Write to memory address 0xFF that will prompt the IQS397 to open a
        communication window.*/
        Wire.write(0xFF);

        /* End the transmission, user decides to STOP or RESTART. */
        Wire.endTransmission(STOP);
    }
}
