/******************************************************************************
 *                                                                            *
 *                                Copyright by                                *
 *                                                                            *
 *                              Azoteq (Pty) Ltd                              *
 *                          Republic of South Africa                          *
 *                                                                            *
 *                           Tel: +27(0)21 863 0033                           *
 *                           E-mail: info@azoteq.com                          *
 *                                                                            *
 * ========================================================================== *
 * @file        iqs397-example-code.ino                                       *
 * @brief       IQS397 example code                                          *
 *                                                                            *
 * @details     This example demonstrates how to write the desired settings   *
 *              to the IQS397 device and how to read button, proximity,       *
 *              haptics, and power-mode events.                               *
 *                                                                            *
 *              Data is displayed over serial communication with the          *
 *              following outputs:                                            *
 *                  - Touch and proximity events                              *
 *                  - Haptics trigger events                                  *
 *                  - Power mode status                                       *
 *                  - Forced communication when 'f' is sent over Serial       *
 *                                                                            *
 * @author      Azoteq PTY Ltd                                                *
 * @version     v1.0                                                          *
 * @date        2026-02-12                                                    *
 ******************************************************************************/

#include <Arduino.h>
#include "src\IQS397\IQS397.h"

/*** Defines ***/
#define DEMO_IQS397_ADDR 0x56
#define DEMO_IQS397_POWER_PIN 4
#define DEMO_IQS397_RDY_PIN 7


/*** Instances ***/
// Create an IQS397 instance. Method calls are defined in the IQS397.h file
// and the method implementation is done in the IQS397.cpp file
IQS397 iqs397;

/*** Global Variables ***/
uint8_t running_display_counter = 0;

void force_comms_and_reset(void); // Method to open a comms window
char read_message(void);          // Method to read data over Serial communication
void print_Heading(void);         // Method to print output headings
void check_power_mode(void);      // Method to check the currennt power mode
void iqs397_ready_interrupt(void);// A method used as an interrupt function

void setup()
{
    /* Start Serial Communication */
    Serial.begin(115200);
    while (!Serial);
    Serial.println("Start Serial communication");

    /* Power On IQS397 */
    pinMode(IQS397_PRODUCT_NUM_0, OUTPUT);
    digitalWrite(DEMO_IQS397_POWER_PIN, LOW);
    delay(200);
    digitalWrite(DEMO_IQS397_POWER_PIN, HIGH);

    /* Initialize the IQS397 with input parameters device address and RDY pin */
    iqs397.begin(DEMO_IQS397_ADDR, DEMO_IQS397_RDY_PIN, iqs397_ready_interrupt);
    delay(2);
}

void loop()
{
    iqs397.run(); // Runs the IQS397

    force_comms_and_reset(); // Function to initialize a force communication window.

    /* Process data read from IQS397 when new data is available (RDY Line Low) */
    if (iqs397.new_data_available)
    {
        print_Heading();
        check_button_flags();
        check_haptics_flags();
        check_power_mode();
        iqs397.new_data_available = false;
    }
}

/* Function to check when the Current power mode of the IQS397 changed */
void check_power_mode(void)
{
    // read current power mode
    switch (iqs397.get_PowerMode())
    {
    case IQS397_NORMAL_POWER:
        Serial.println("\tNORMAL POWER");
        break;
    case IQS397_ULP:
        Serial.println("\tULP");
        break;
    default:
        break;
    }
}

void print_Heading(void)
{
    /* Check if it is necessary to display the heading
       Display heading after every 20 iterations*/
    if (running_display_counter == 20 || running_display_counter == 0)
    {
        /* Print the extra heading for the serial feedback */
        Serial.println("\nButton State:\tHaptics Trigger:\tPower Mode:");

        running_display_counter = 1;
    }

    running_display_counter++;
}

/* Function to check if haptics is triggered */
void check_haptics_flags(void)
{
    /* check if the haptics event bit is set (triggered) for channel 1 */
    if (iqs397.haptics_triggerEventState())
    {
        Serial.print("Haptics\t\t");
    }
    /* None state if the haptics event bit is not set (triggered) */
    else
    {
        Serial.print("--\t\t");
    }
}

/* Function to check the Button Active state of the self-capacitance button */
void check_button_flags(void)
{
    /* check if the button active state bit is set for channel 1 */
    if (iqs397.button_touchState())
    {
        Serial.print("Touch\t\t");
    }
    else if (iqs397.button_proximityState())
    {
        Serial.print("Proximity\t");
    }
    /* None state if neither the enter nor exit bit was set to 1 */
    else
    {
        Serial.print("--\t\t");
    }
}

/* Force the IQS397 to open a RDY window and read the current state of the
device or request a software reset */
void force_comms_and_reset(void)
{
    char message = read_message();

    /* If an 'f' was received over serial, open a forced communication window and
    print the new data received */
    if (message == 'f')
    {
        iqs397.force_I2C_communication(); // prompt the IQS397
    }

    /* If an 'r' was received over serial, request a software(SW) reset */
    if (message == 'r')
    {
        Serial.println("Software Reset Requested!");
        iqs397.force_I2C_communication(); // Request a RDY window
        iqs397.do_soft_reset = true;
        running_display_counter = 0;
    }
}

/* Read message sent over serial communication */
char read_message(void)
{
    while (Serial.available())
    {
        if (Serial.available() > 0)
        {
            return Serial.read();
        }
    }

    return '\n';
}

/* A method used as an interrupt function. Only activated when a falling-
edge interrupt is seen on the correct Arduino interrupt pin.*/
void iqs397_ready_interrupt(void)
{
    iqs397.ready_interrupt();
}
